; Command level protocol
;
;	Load:
;	1.	Computer sends 0,"name",0
;
;	2.	Computer then reads one byte.
;		If it's not 0 then computer reads that many
;		bytes into memory and loops to 2.
;
;	3.	Computer reads one more byte.
;			If it's 0 then load was OK.
;
;			1-15 are drive low-level errors
;			62 is "file not found"
;
;
;	Save:
;	1.	Computer sends 1,"name",0
;
;	2.	Computer then sends 252 bytes.
;
;	------------------------------------------------------------------
;
; Signal level protocol
;
;	In idle state both C & D are 0
;
;	Computer wants to send command:
;		set D=1
;		waits for C=1
;		set D=0
;
;	Drive waits for command:
;		enable interrupts (stop motor, 1581 flush)
;		wait D=1
;		disable interrupts
;		set C=1
;		wait for D=0
;
;
;	Now we are in a state where drive keeps C=1 to tell it's busy.
;
;	----
;
;	Computer sends byte:
;		wait for C=0
;		for bit=0 to 7 (min. 18 us per bit)
;			set data bit, toggle clock
;		set C=0, D=0
;
;	Drive reads byte:
;		set C=0
;		for bit=0 to 7
;			wait for C toggle
;			read data bit
;		set C=1
;
;	----
;
;	Computer reads byte:
;		wait for C=0
;		for bit=0 to 7 (min. 19 usec per bit)
;			read data bit
;			toggle clock
;
;	Drive sends byte:
;		D = bit0
;		set C=0
;		for bit=1 to 7
;			wait for C toggle
;			set data bit
;		wait for C=0
;		set C=1, D=0
;
;	When command is completed drive sets C=0 and we are in idle state


	processor 6502
	seg	code

;	time taken to load XeO3 teaser (60673 bytes):
;
;	1541:	155 seconds without loader
;		 40 seconds with interleave 16
;
;	1581:	115 seconds without loader
;		 18 seconds with loader

;	these two control kernal compatibility

LOAD_ADDRESS_OVERRIDE	= 1		; can specify load address with
					; SA=0, y/x=load address

RETURN_END_ADDRESS	= 1		; return end address in y/x


l_addr	= $9d
SA	= $ad
devnum	= $ae
tmp	= $d0				; 5 bytes

setlfs	= $ffba
setnam	= $ffbd
open	= $ffc0
close	= $ffc3
chkin	= $ffc6
chkout	= $ffc9
clrchn	= $ffcc
chrin	= $ffcf
chrout	= $ffd2


;	Dout *MUST* be 1, Cin *MUST* be 64, Din *MUST* be 128

iec	= $01				; I/O
Dout	= 1
Cout	= 2
Atn	= 4
Cin	= 64
Din	= 128


	org	$1001

	dc.w	$100a,$06bf		; BASIC line "1727 sys4112"
	dc.b	$9e,"4112",0
	dc.b	0,0
	dc.b	"TNT"

	subroutine
Start
	ldx	devnum			; assume device #8 if no device
	bne	.1
	ldx	#8
.1	lda	#15
	tay
	jsr	setlfs
	lda	#0
	jsr	setnam
	jsr	open			; open 15,(8),15

	jsr	DriveID			; check & patch for 154x/157x/1581
	bcc	.cont			; continue if ok

	ldx	#15			; else exit without installing
	jsr	close			; the loader
	rts

.found	dc.b	" FOUND",13,0

.cont	lda	MRdata+4,x		; print drive type
	jsr	chrout
	inx
	txa
	and	#3
	bne	.cont

	tay
.c1	lda	.found,y
	beq	.c2
	jsr	chrout
	iny
	bne	.c1

.c2	lda	#0			; send drive code
	sta	tmp
	lda	#5
	sta	tmp+1
	lda	#<code1541
	sta	tmp+2
	lda	#>code1541
	sta	tmp+3

	lda	#16			; 16*32=512 bytes
	sta	tmp+4

.mw	ldx	#15			; 32-byte chunk
	jsr	chkout

	lda	#"M"			; "M-W",<dest>,32
	jsr	chrout
	lda	#"-"
	jsr	chrout
	lda	#"W"
	jsr	chrout

	lda	tmp
	jsr	chrout
	lda	tmp+1
	jsr	chrout
	lda	#$20
	jsr	chrout

	ldy	#0			; data bytes
.mw2	lda	(tmp+2),y
	jsr	chrout
	iny
	cpy	#$20
	bne	.mw2

	clc				; bump dest address
	tya
	adc	tmp
	sta	tmp
	bcc	.mw3
	inc	tmp+1
.mw3
	clc
	tya				; bump source address
	adc	tmp+2
	sta	tmp+2
	bcc	.mw3b
	inc	tmp+3
.mw3b
	lda	#13
	jsr	$ffd2
	jsr	clrchn			; send it for real

	dec	tmp+4			; until 512 bytes sent
	bne	.mw


	ldx	#15
	jsr	chkout
	lda	#"U"
	jsr	chrout
	lda	#"3"
	jsr	chrout
	jsr	clrchn			; "U3", jmp $0500

	lda	iec			; clk=dat=atn=0
	and	#255-Cout-Dout-Atn
	sta	iec

;	Note: there must be a small delay between the above
;	and the first disk access. Normally that's taken care
;	of by clear/copy loops below.

	lda	#0			; clear key definitions
	ldx	#10
.zk	sta	$055d-1,x
	dex
	bne	.zk

.cc	lda	patch,x			; copy resident load/save
	sta	$0568,x			; routine to $0568
;	lda	patch+256,x
;	sta	$0668,x
	inx
	bne	.cc

	lda	#<Load			; point load/save to our code
	ldx	#>Load
	sta	$032e
	stx	$032f

	lda	#<Save
	ldx	#>Save
	sta	$0330
	stx	$0331

	rts

;	------------------------------------------------------------------

;	find out which drive we are talking to

;	we use reset/IRQ vectors at $fffc-$ffff
;
;		fffc fffe
;		---- ----
;		eaa0 fe67	1541, 1541-II, 1570, 1571, 1571cr
;		af24 ff03	1581
;		e9b5 f195	1551
;
;	1541/1571 IRQ routine at $fe67:
;
;		48 8a 48 98	1541, 1541-II
;		6c a9 02 98	1570, 1571, 1571cr

	subroutine
DriveID
	ldy	#0			; read $fffc-$ffff
	jsr	MR4

	ldx	#0			; 1581?
	jsr	.id
	beq	.1581
	ldx	#8			; 1551?
	jsr	.id
	beq	.1551
	ldx	#16			; 1541/157x?
	jsr	.id
	bne	.unkn

;	additional check for 154x/157x

	ldy	#2			; read $fe67-$fe6a
	jsr	MR4

	ldx	#24			; 1541?
	jsr	.id
	beq	.1541
	ldx	#32			; 1571?
	jsr	.id
	beq	.1571

.1551
.unkn	sec
	rts

.1581	jsr	Fix1581			; patch some addresses
	ldx	#4
.1541
.1571	clc
	rts

.id	ldy	#0
.i1	lda	MRdata,y
	cmp	MRdata+4,x
	bne	.ix
	inx
	iny
	cpy	#4
	bne	.i1
.ix	rts


MRaddr	dc.w	$fffc,$fe67
MRdata	ds.b	4

	dc.w	$af24,$ff03
	dc.b	"1581"
	dc.w	$e9b5,$f195
	dc.b	"1551"
	dc.w	$eaa0,$fe67
	dc.b	"15x1"

	hex	48 8a 48 98
	dc.b	"1541"
	hex	6c a9 02 98
	dc.b	"1571"

;	M-R four bytes from specified address

MR4	ldx	#15
	jsr	chkout

	lda	#"M"
	jsr	chrout
	lda	#"-"
	jsr	chrout
	lda	#"R"
	jsr	chrout

	lda	MRaddr,y
	jsr	chrout
	lda	MRaddr+1,y
	jsr	chrout
	lda	#4
	jsr	chrout
	jsr	clrchn			; "m-r" ...

	ldx	#15
	jsr	chkin

	ldy	#0
.mr1	jsr	chrin
	sta	MRdata,y
	iny
	cpy	#4
	bcc	.mr1

	jsr	clrchn
	rts
	
;	------------------------------------------------------------------

;	changes for 1581

;		IEC port address $1800 -> $4001, 
;		LED port address $1c00 -> $4000,
;		job code, track & sector,
;		LED bit, and
;		directory track & sector

	subroutine
Fix1581
	lda	#<code1541
	ldy	#>code1541
	sta	tmp
	sty	tmp+1

.1	ldy	#0
	lda	(tmp),y
	bne	.3
	iny
	lda	(tmp),y
	cmp	#$18
	beq	.2			; 1800 -> 4001
	cmp	#$1c
	bne	.3
	clc				; 1c00 -> 4000

.2	lda	#$40
	sta	(tmp),y
	bcc	.3
	tya
	dey
	sta	(tmp),y

.3	inc	tmp
	bne	.4
	inc	tmp+1

.4	lda	tmp
	cmp	#<_code1541
	lda	tmp+1
	sbc	#>_code1541
	bcc	.1			; tmp < _code1541

;	fix remaining addresses

	lda	#2			; job code, track & sector
	sta	code1541+mod81_1-Load41+1
	sta	code1541+mod81_2-Load41+1
	ldx	#11
	stx	code1541+mod81_t-Load41+1
	inx
	stx	code1541+mod81_s-Load41+1

	lda	#$20			; LED bit
	sta	code1541+mod81_l1-Load41+1
	eor	#$ff
	sta	code1541+mod81_l2-Load41+1

;	fix directory track/sector

	ldx	#4
.5	lda	.ddat-1,x
	sta	code1541+mod81_d-Load41,x
	dex
	bne	.5

	rts

.ddat	dc.b	3			; sector
	ldx	$022b			; track





;	------------------------------------------------------------------
;	resident loader code
;	------------------------------------------------------------------

	subroutine
patch
	rorg	$0568

;	first byte from computer to 1541:
;	0 - load
;	1 - save
;
;	then comes filename which ends with $00

Load
 IF	LOAD_ADDRESS_OVERRIDE
	stx	l_addr			; assume x/y load address
	sty	l_addr+1
 ENDIF

	ldx	#0			; load
	jsr	SendName		; also goes to slow mode
	pha				; store $ff13 value

	jsr	Rx16			; #bytes to transfer
	beq	.5			; end

;	clc				; Rx16 clears carry for us
	sbc	#2-1

 IF	LOAD_ADDRESS_OVERRIDE
	sta 	.1+1

	jsr	Rx16
	tax
	jsr	Rx16
	ldy	SA			; load address given in x/y?
	beq	.1			; yes
	stx	l_addr
	sta	l_addr+1

.1	ldx	#$fc
 ELSE
	tax

	jsr	Rx16
	sta	l_addr
	jsr	Rx16
	sta	l_addr+1
 ENDIF
	bcc	.3			; jmp

.2	jsr	Rx16			; #bytes
	tax
	beq	.5			; end

.3	ldy	#0
.4	jsr	Rx16
	sta	(l_addr),y
	iny
	dex
	bne	.4

;	clc				; Rx16 did this
	tya				; bump dest
	adc	l_addr
	sta	l_addr
	bcc	.2
	inc	l_addr+1
	bcs	.2			; jmp

.5	jsr	Rx16			; error info
	cmp	#2			; Fc=1 if >=2

 IF RETURN_END_ADDRESS
	ldx	l_addr
	ldy	l_addr+1
 ENDIF

.x	pla				; restore slow/fast mode
	sta	$ff13
	rts

;	----

Save	ldx	#1			; save
	jsr	Tx16
	jsr	SendName
	pha

	ldy	#0
.s1	php
	sei
	sta	$ff3f			; RAM in
	lda	($b2),y			; data byte
	sta	$ff3e			; ROM in
	plp
	jsr	Tx16
	iny
	cpy	#252			; always 252 bytes
	bcc	.s1

	clc				; exit via Load routine
	bcc	.x

;	----

	subroutine
SendName
	lda	$ff13			; slow mode
	pha
	ora	#$02
	sta	$ff13

;	command hanshake

	lda	iec			; D=1
	ora	#Dout
	sta	iec

	bit	iec			; wait C=1
	bvs	*-3

	and	#255-Dout		; D=0
	sta	iec

;	send command code and file name
	
	txa
	jsr	Tx16

	ldy	#0
.txname	php
	sei
	sta	$ff3f			; RAM in
	lda	($af),y			; filename char
	sta	$ff3e			; ROM in
	plp
	jsr	Tx16
	iny
	tya
	eor	$ab			; clears A too
	bne	.txname
;	lda	#0
	jsr	Tx16

	ldx	#0
	inx
	bne	*-1

	pla				; return $ff13
	rts

;	----
;	send byte to drive

	subroutine
Tx16	sec				; prepare end bit
	ror				; and first data bit

.w	bit	iec
	bvc	.w

.1	pha				;  3
	lda	#Cout			;  5
	eor	iec			;  8		toggle clk
	and	#255-Dout		; 10		Fc->dat
	adc	#0			; 12
	sta	iec			; 15
	pla				; 19
	lsr				; 21
	bne	.1			; 23 24

	lda	iec			; 26
	and	#255-Cout-Dout-Atn	; 28
	sta	iec			; 31		14 cycles to grab last bit
	rts

;	----
;	read byte from drive

	subroutine
Rx16	lda	#$7f			; prepare end bit
	sta	tmp+3

	bit	iec			; wait until clock is released
	bvc	*-2			; (1541 sends first data bit with it)

.1	lda	#Cout			;  2		toggle clock, read data
	eor	iec			;  5
	sta	iec			;  8
	asl				; 10		Din -> Fc
	ror	tmp+3			; 15		Fc -> byte
	bcs	.1			; 18

	lda	tmp+3
	rts

	rend


;	------------------------------------------------------------------
;	drive code
;
;	same code is used for 1541/157x/1581, but 1581 has some
;	addresses/values changed
;	------------------------------------------------------------------


Din_d	= 1		; i
Dout_d	= 2		; o
Cin_d	= 4		; i
Cout_d	= 8		; o
Aout_d	= 16		; o
Dnum_d	= 96		; ii	1541
fsdir_d	= 32		; o	1581
wpin_d	= 64		; i	1581
Ain_d	= 128		; i

buf	= $0300
iec_d	= $1800		; data port	1581: 4001
ledp	= $1c00		; led port	1581: 4000
led	= $08		; led bit	1581: 20

job	= 0		; job code	1581: 2
track	= 6		; job track	1581: 11
sector	= 7		; job sector	1581: 12

	subroutine

code1541
	rorg	$0500
Load41
	sei
	lda	#Cout_d			; clk=1, dat=0 for "not ready"
	sta	iec_d

;	compare value for clock
;	when we send data Cout=0, Cin toggles, Dout=?, Din=0
;	when we receive data Cout=0, Cin toggles, Dout=0, Din=?

	lda	iec_d
	and	#$70			; clk=0, dat=0
	ora	#$03			; data=1
	sta	Tx41x+1			; compare value for xfer

;	compare values for data read

	eor	#05^03			; clk=1
	sta	g41c1+1
	eor	#01^05			; clk=0
	sta	g41c2+1

.loop
	lda	ledp			; LED off
mod81_l2
	and	#255-led
	sta	ledp

;	wait for command handshake

	cli				; allow drive to stop, also
					; allows 1581 flush cache
	lda	#Din_d
.wcmd	bit	iec_d
	bmi	.exit			; ATN? reset
	beq	.wcmd

;	disable interrupts and ack with clk=1

	sei
	ldx	#Cout_d
	stx	iec_d

;	now wait until dat=0 again

	bit	iec_d
	bne	*-3

	jsr	Rx41			; load/save info
	pha
	jsr	GetName			; and name

	lda	ledp			; LED on
mod81_l1
	ora	#led
	sta	ledp

	jsr	FindFile
	pla
	bne	.save			; "save" even if no file
	bcs	.err62			; file not found


.load

.l1	jsr	ReadTS			; read block
	bcs	.error

	lda	buf+1			; last byte to send
	ldx	buf			; if last block
	beq	.l2
	lda	#$ff			; otherwise send 254 bytes
.l2	sta	.l5+1			; 

	sbc	#1-1
	jsr	Tx41x			; #bytes to follow

	ldy	#2-1			; send data from buffer
.l3	iny
	sty	.l4+1
	lda	buf,y
	jsr	Tx41
.l4	ldy	#0
.l5	cpy	#0
	bne	.l3

	ldy	buf+1			; next t/s
	ldx	buf
	bne	.l1			; not end yet

	txa				; =0, no error
	beq	.error			; jmp

.save	bcs	.s1			; no save if no file
	jsr	ReadTS			; no save if read error
.s1	php

	lda	#4			; we receive 252 bytes
	sta	.s3+1			; in any case
.s2	jsr	Rx41
.s3	sta	buf+4
	inc	.s3+1
	bne	.s2 

	plp				; should we really save it?
	bcs	.sx			; no
	jsr	WriteTS
.sx	jmp	.loop			; back to command loop

.exit	jmp	($fffc)			; reset

.err62	lda	#62
.error	pha
	lda	#0			; error/end coming up
	jsr	Tx41x
	pla				; error code
	jsr	Tx41x
	jmp	.loop

;	----


;	get filename from computer

namebuf	= $0200

	subroutine
GetName	lda	#0
	sta	.2+1
.1	jsr	Rx41
.2	sta	$0200
	inc	.2+1
	tay
	bne	.1
	rts

;	find file in directory

	subroutine
FindFile

mod81_d	ldy	#1			; "ldy #3; ldx $22b" if 1581
	ldx	#18
	nop

.1	jsr	ReadTS
	bcs	.error

	ldy	#0
.2	lda	buf+2,y			; file type
	and	#$87
	cmp	#$82
	bne	.next			; not prg

	tya
	adc	#5-1
	sta	.3+1

	ldx	#0
.3	lda	buf+5,x			; filename char
	cmp	#$a0
	beq	.end
	cmp	namebuf,x		; wanted char
	bne	.next
	inx
	cpx	#16
	bcc	.3

.found	ldx	buf+3,y			; track
	lda	buf+4,y			; sector
	tay
	clc
	rts

.end	lda	namebuf,x		; match if wanted name ended as well
	beq	.found

.next	clc
	tya
	adc	#32
	tay
	bcc	.2

	ldy	buf+1			; next t/s
	ldx	buf
	bne	.1

.error
;	sec
	rts

	subroutine
ReadTS
mod81_t	stx	track
mod81_s	sty	sector

	lda	#$80			; READ job
	dc.b	$2c
WriteTS
	lda	#$90			; WRITE job
	sta	.1+1

;	x=state
;		 2 - read/write
;		 1 - bump
;		 0 - read/write
;		-1 - exit

	ldx	#2
.1	lda	#$80			; READ/WRITE
	dc.b	$2c
.2	lda	#$c0			; BUMP


mod81_1	sta	job			; start job
	cli
.3
mod81_2	lda	job			; wait until it finishes
	bmi	.3

	ldy	$16			; copy real disk ID
	sty	$12			; to wanted ID
	ldy	$17			; so disk can be
	sty	$13			; changed

	cmp	#2			; Fc=1 if error

	dex
	bmi	.x			; exit if r/w after bump done
	beq	.1			; don't exit after bump
	bcs	.2			; retry with bump if error

.x	sei				; back to transfer loop
	rts

;	----

;	read byte from computer
;
;	not unrolled as max. size is 252 bytes

	subroutine
Rx41	lda	#$80			; end bit
	sta	iec_d			; clk=0 (data=0) for "we're ready"
	ldx	Tx41x+1

.1	cpx	iec_d			;  4	wait for clk=1
	bcs	.1			;  6
	ldy	iec_d			; 10	read data bit
g41c1	cpy	#$05			; 12	-> carry
	ror				; 14	-> a

.2	cpx	iec_d			;  4	wait for clk=0
	bcc	.2			;  6
	ldy	iec_d			; 10
g41c2	cpy	#$01			; 12
	ror				; 14
	bcc	.1			; 17

	ldy	#Cout_d			; clk=1 for "not ready"
	sty	iec_d
	rts

;	----

;	send byte to computer

;	unrolled to keep up with 18 (slow) cycle routine at computer

	MAC	s1
	lsr				;  2	prepare data bit in y
	ldy	#Dout_d			;  4
	bcc	*+4			;  6
	ldy	#0			;  8
	cpx	iec_d			; 12	wait for clk=0
	bcc	*-3			; 14
	sty	iec_d			; 18
	ENDM				;	1+2+2+3+2+3=13 bytes

	MAC	s2			; same with other clock edge
	lsr
	ldy	#Dout_d
	bcc	*+4
	ldy	#0
	cpx	iec_d			;	wait for clk=1
	bcs	*-3
	sty	iec_d
	ENDM



Tx41x
	ldx	#0			; self-modified
Tx41
	lsr
	ldy	#2
	bcc	*+4
	ldy	#0
	sty	iec_d			;	dat=?, clk=0 for "we're ready"

	s2				;	wait clk=1
	s1
	s2				
	s1
	s2
	s1
	s2				;	wait clk=1

;	tya				; 	keep data, clk=1 for "not ready"
;	ora	#Cout_d
	ldy	#Cout_d
	cpx	iec_d			;	wait for clk=0
	bcc	*-3
	sty	iec_d			;	clk=1 "not ready"
	rts

	rend
_code1541
